chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "analyze") {
    const keyphrase = request.keyphrase.toLowerCase();
    const results = analyzeSEO(keyphrase);
    sendResponse(results);
  }
});

function analyzeSEO(keyphrase) {
  const report = [];
  const doc = document;
  const bodyText = document.body.innerText;
  const wordCount = bodyText.trim().split(/\s+/).length;
  const hostname = window.location.hostname;
  
  // Helper to add result
  const addResult = (passed, message, isCritical = false) => {
    report.push({ passed, message, isCritical });
  };

  // 1. Outbound links
  const links = Array.from(doc.getElementsByTagName('a'));
  const outbound = links.filter(a => a.hostname && a.hostname !== hostname).length;
  addResult(
    outbound > 0,
    outbound > 0 ? "Outbound links: Good job!" : "Outbound links: No outbound links found."
  );

  // 2. Images: Keyphrase in Alt
  const images = Array.from(doc.getElementsByTagName('img'));
  const imagesWithKeyphrase = images.filter(img => img.alt.toLowerCase().includes(keyphrase)).length;
  addResult(
    imagesWithKeyphrase > 0,
    imagesWithKeyphrase > 0 ? "Keyphrase in image alt attributes: Good job!" : "Keyphrase in image alt attributes: Focus keyphrase not found in any image alt tags."
  );

  // 3a. Images: Missing Alt Tags (NEW FEATURE)
  // Check for empty alt or missing alt attribute
  const imagesMissingAlt = images.filter(img => !img.hasAttribute('alt') || img.alt.trim() === "").length;
  addResult(
    imagesMissingAlt === 0,
    imagesMissingAlt === 0 
      ? "Image Alt Attributes: All images have alt attributes. Great for accessibility!" 
      : `Image Alt Attributes: ${imagesMissingAlt} images are missing alt text. This hurts SEO and accessibility.`,
    true // Critical error
  );

  // 3b. Images Count
  addResult(
    images.length > 0,
    images.length > 0 ? `Images: Found ${images.length} images on page.` : "Images: Consider adding images to your page."
  );

  // 4. Internal links
  const internal = links.filter(a => a.hostname === hostname).length;
  addResult(
    internal > 0,
    internal > 0 ? "Internal links: Good job!" : "Internal links: No internal links found."
  );

  // 5. Keyphrase in introduction
  const firstP = doc.querySelector('p');
  const introHasKeyphrase = firstP && firstP.innerText.toLowerCase().includes(keyphrase);
  addResult(
    introHasKeyphrase,
    introHasKeyphrase ? "Keyphrase in introduction: Well done!" : "Keyphrase in introduction: Not found in the first paragraph."
  );

  // 6. Keyphrase density
  const regex = new RegExp(escapeRegExp(keyphrase), 'gi');
  const matches = (bodyText.match(regex) || []).length;
  const density = wordCount > 0 ? ((matches / wordCount) * 100).toFixed(2) : 0;
  addResult(
    matches > 0,
    `Keyphrase density: Found ${matches} times (${density}%).`
  );

  // 7. SEO Title
  const pageTitle = doc.title.toLowerCase();
  const startsWith = pageTitle.startsWith(keyphrase);
  const titleHas = pageTitle.includes(keyphrase);
  addResult(
    titleHas,
    startsWith 
      ? "Keyphrase in SEO title: Perfect match at the beginning." 
      : (titleHas ? "Keyphrase in SEO title: Found, but not at the start." : "Keyphrase in SEO title: Not found."),
    true // Critical
  );

  // 8. Keyphrase length
  const kpLen = keyphrase.split(' ').length;
  addResult(
    kpLen > 0 && kpLen < 10,
    "Keyphrase length: Good length."
  );

  // 9. Meta Description
  const metaDescTag = doc.querySelector('meta[name="description"]');
  const metaDesc = metaDescTag ? metaDescTag.content.toLowerCase() : "";
  const metaHas = metaDesc.includes(keyphrase);
  addResult(
    metaHas,
    metaHas ? "Keyphrase in meta description: Found." : "Keyphrase in meta description: Not found.",
    true
  );

  // 10. Meta Description Length
  const mdLen = metaDesc.length;
  addResult(
    mdLen >= 120 && mdLen <= 160,
    (mdLen >= 120 && mdLen <= 160) ? "Meta description length: Good." : `Meta description length: ${mdLen} chars (Ideal: 120-160).`
  );

  // 12. Single H1
  const h1Count = doc.getElementsByTagName('h1').length;
  addResult(
    h1Count === 1,
    h1Count === 1 ? "Single H1: Good job!" : `Single H1: Found ${h1Count} H1 tags. (Should be exactly 1).`,
    true
  );

  // 13. Slug
  const slug = window.location.pathname.toLowerCase();
  const slugKey = keyphrase.replace(/\s+/g, '-');
  addResult(
    slug.includes(slugKey) || slug.includes(keyphrase),
    "Keyphrase in slug: URL looks good."
  );

  // 14. Subheadings
  const subs = [...Array.from(doc.getElementsByTagName('h2')), ...Array.from(doc.getElementsByTagName('h3'))];
  const subMatch = subs.filter(h => h.innerText.toLowerCase().includes(keyphrase)).length;
  addResult(
    subMatch > 0,
    subMatch > 0 ? `Keyphrase in subheadings: Found in ${subMatch} headings.` : "Keyphrase in subheadings: Not found in H2 or H3."
  );

  // 15. Competing Links
  const competing = links.filter(a => a.innerText.toLowerCase().trim() === keyphrase).length;
  addResult(
    competing === 0,
    competing === 0 ? "Competing links: Clean." : "Competing links: You are linking out with your exact focus keyword."
  );

  // 16. Text Length
  addResult(
    wordCount >= 300,
    wordCount >= 300 ? `Text length: ${wordCount} words (Good).` : `Text length: ${wordCount} words. Recommend > 300.`
  );

  // 17. Title Width
  const titleLen = doc.title.length;
  addResult(
    titleLen >= 30 && titleLen <= 60,
    (titleLen >= 30 && titleLen <= 60) ? "SEO title width: Good." : "SEO title width: Too short or too long."
  );

  return report;
}

function escapeRegExp(string) {
  return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}